/**
 * \file: config.c
 *
 * \version: $Id:$
 *
 * \release: $Name:$
 *
 * \component: Secure Data Container
 * \brief : Functions to access the configuration of the sdc
 *
 * \author: Norbert Uetrecht (nuetrecht@de.adit-jv.com)
 *      Christoph Gellner (cgellner@de.adit-jv.com)
 *
 * \copyright (c) 2014 Advanced Driver Information Technology.
 * This code is developed by Advanced Driver Information Technology.
 * Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
 * All rights reserved.
 *
 *
 ***********************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <stdbool.h>

#include <private/sdc_advanced.h>
#include <sdc.h>
#include "config.h"

/* External available interface as defined in sdc.h */
sdc_error_t sdc_config_file_lookup(char **val, const char *lookup)
{
    char *ret, *value = NULL, *key = NULL;
    FILE *fp;
    size_t len;
    char *buf, *pos;

    enum {
        KM_STATE_START,
        KM_STATE_IN_KEY,
        KM_STATE_IN_VALUE,
        KM_STATE_IN_COMMENT,
        KM_STATE_SKIP_VALUE
    } state;
    bool found = false;
    sdc_error_t err;

    if ((!val) || (!lookup))
        return SDC_INVALID_PARAMETER;

    *val = NULL;

    fp = fopen(CONFIG_PATH, "r");
    if (fp == NULL) {
        return SDC_CONFIG_ACCESS_FAILED;
    }

    /* Don't know how long a line will be, so find file length */
    fseek(fp, 0, SEEK_END);
    len = ftell(fp);
    fseek(fp, 0, SEEK_SET);

    buf = malloc(len);
    if (buf == NULL) {
        fclose(fp);
        return SDC_NO_MEM;
    }

    if (fread(buf, 1, len, fp) != len) {
        fclose(fp);
        free(buf);
        return SDC_CONFIG_ACCESS_FAILED;
    }

    fclose(fp);

    /* Parse config file */
    ret = NULL;
    pos = buf;
    state = KM_STATE_START;
    err = SDC_CONFIG_MISSING;
    while (pos < buf + len && (!found)) {
        switch (state) {
        case KM_STATE_START:
            if (!isspace(*pos)) {
                if (*pos == '#') {
                    state = KM_STATE_IN_COMMENT;
                } else {
                    state = KM_STATE_IN_KEY;
                    key = pos;
                }
            }
            break;
        case KM_STATE_IN_KEY:
            if (*pos == '=' && key != NULL) {
                *pos='\0';
                if (strcmp(key, lookup) == 0) {
                    state = KM_STATE_IN_VALUE;
                    value = pos + 1;
                } else {
                    state = KM_STATE_SKIP_VALUE;
                }
            }
            break;
        case KM_STATE_IN_VALUE:
            if (*pos == '\n' && value != NULL) {
                ret = strndup(value, pos - value);
                if (ret == NULL) {
                    free(buf);
                    return SDC_NO_MEM;
                }
                err = SDC_OK;
                found = true;
            }
            break;
        case KM_STATE_SKIP_VALUE:
        case KM_STATE_IN_COMMENT:
            if (*pos == '\n') {
                state = KM_STATE_START;
            }
            break;
        default:
            state = KM_STATE_START;
            break;
        }
        pos++;
    }

    free(buf);
    *val = ret;
    return err;
}

sdc_error_t sdc_config_is_absolute_path(const char *path)
{
    if (!path)
        return SDC_CONFIG_INVALID;

    if (*path != '/')
        return SDC_CONFIG_INVALID;

    return SDC_OK;
}

sdc_error_t sdc_config_file_lookup_absolute_path(char **val, const char *lookup)
{
    sdc_error_t err;

    err = sdc_config_file_lookup(val, lookup);
    if (err != SDC_OK)
        return err;

    err = sdc_config_is_absolute_path(*val);
    if (err != SDC_OK) {
        free(*val);
        *val = NULL;
        return err;
    }

    return SDC_OK;
}
